<?php
require __DIR__ . '/../src/bootstrap.php';
$pdo = db();

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);

// Simple routing
if ($method === 'POST' && $uri === '/submit') {
    // Handle form submission
    $appointmentId = (int)($_POST['appointment_id'] ?? 0);
    if (!$appointmentId) {
        http_response_code(400);
        echo 'Invalid appointment.';
        exit;
    }

    // Ensure appointment exists and not already completed
    $stmt = $pdo->prepare('SELECT a.id, a.customer_id, lf.id AS lf_id FROM appointments a LEFT JOIN liability_forms lf ON lf.appointment_id = a.id WHERE a.id = ?');
    $stmt->execute([$appointmentId]);
    $row = $stmt->fetch();
    if (!$row) {
        http_response_code(404);
        echo 'Appointment not found.';
        exit;
    }
    if (!empty($row['lf_id'])) {
        echo 'Form already submitted.';
        exit;
    }

    $fields = [
        'device_make' => $_POST['device_make'] ?? '',
        'device_model' => $_POST['device_model'] ?? '',
        'device_password' => $_POST['device_password'] ?? null,
        'problem_description' => $_POST['problem_description'] ?? '',
        'accessories_left' => $_POST['accessories_left'] ?? '',
        'condition_notes' => $_POST['condition_notes'] ?? '',
        'contact_preference' => $_POST['contact_preference'] ?? 'contact_ok',
        'consent_data_backup' => isset($_POST['consent_data_backup']) ? 1 : 0,
        'consent_password_handling' => isset($_POST['consent_password_handling']) ? 1 : 0,
        'signature_name' => $_POST['signature_name'] ?? '',
        'signature_ip' => $_SERVER['REMOTE_ADDR'] ?? null,
    ];

    // Basic validation
    if (!$fields['signature_name']) {
        http_response_code(422);
        echo 'Signature required.';
        exit;
    }
    if (!in_array($fields['contact_preference'], ['no_contact','contact_ok'], true)) {
        $fields['contact_preference'] = 'contact_ok';
    }

    $pdo->beginTransaction();
    try {
        $stmt = $pdo->prepare('INSERT INTO liability_forms (
            appointment_id, customer_id, device_make, device_model, device_password, problem_description, accessories_left, condition_notes, contact_preference,
            consent_data_backup, consent_password_handling, signature_name, signature_ip, signature_timestamp, version, created_at, updated_at
        ) VALUES (
            :appointment_id, :customer_id, :device_make, :device_model, :device_password, :problem_description, :accessories_left, :condition_notes, :contact_preference,
            :consent_data_backup, :consent_password_handling, :signature_name, :signature_ip, NOW(), 1, NOW(), NOW()
        )');
        $stmt->execute([
            ':appointment_id' => $row['id'],
            ':customer_id' => $row['customer_id'],
            ':device_make' => $fields['device_make'],
            ':device_model' => $fields['device_model'],
            ':device_password' => $fields['device_password'] ?: null,
            ':problem_description' => $fields['problem_description'],
            ':accessories_left' => $fields['accessories_left'],
            ':condition_notes' => $fields['condition_notes'],
            ':contact_preference' => $fields['contact_preference'],
            ':consent_data_backup' => $fields['consent_data_backup'],
            ':consent_password_handling' => $fields['consent_password_handling'],
            ':signature_name' => $fields['signature_name'],
            ':signature_ip' => $fields['signature_ip'],
        ]);

        // Optional: mark appointment status
        $pdo->prepare('UPDATE appointments SET status = "liability_complete", updated_at = NOW() WHERE id = ?')->execute([$row['id']]);

        $pdo->commit();
    } catch (Throwable $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo 'Error saving form.';
        exit;
    }

    echo '<h2>Thank you! Your liability form has been submitted.</h2>';
    exit;
}

if ($method === 'POST' && $uri === '/api/mint-link') {
    // Require API key in X-API-Key header
    $allowedKeys = app_config('api_keys') ?: [];
    $providedKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
    if (!$providedKey || !in_array($providedKey, $allowedKeys, true)) {
        json_response(['error' => 'Unauthorized'], 401);
    }
    // Expected from Power Automate: appointmentId
    $input = json_decode(file_get_contents('php://input'), true) ?? [];
    $appointmentId = (int)($input['appointmentId'] ?? 0);
    if (!$appointmentId) {
        json_response(['error' => 'appointmentId required'], 400);
    }
    // Ensure appointment exists
    $stmt = $pdo->prepare('SELECT id FROM appointments WHERE id = ?');
    $stmt->execute([$appointmentId]);
    if (!$stmt->fetch()) {
        json_response(['error' => 'Appointment not found'], 404);
    }

    $exp = time() + 7 * 24 * 60 * 60; // 7 days
    $params = ['a' => $appointmentId, 'exp' => $exp];
    $sig = hmac_sign($params);

    $base = app_config('base_url');
    $url = $base . '/?a=' . urlencode((string)$appointmentId) . '&exp=' . $exp . '&sig=' . $sig;
    json_response(['url' => $url]);
}

// GET form display at '/'
if ($method === 'GET') {
    $a = isset($_GET['a']) ? (int)$_GET['a'] : 0;
    $exp = isset($_GET['exp']) ? (int)$_GET['exp'] : 0;
    $sig = $_GET['sig'] ?? '';

    if (!$a || !$exp || !$sig || !verify_signed_params(['a' => $a, 'exp' => $exp, 'sig' => $sig], $sig)) {
        http_response_code(400);
        echo 'Invalid or expired link.';
        exit;
    }

    // Load appointment + customer
    $stmt = $pdo->prepare('SELECT a.id as appointment_id, a.scheduled_start, a.scheduled_end, a.status, c.id as customer_id, c.first_name, c.last_name, c.email, c.phone
                           FROM appointments a JOIN customers c ON c.id = a.customer_id WHERE a.id = ?');
    $stmt->execute([$a]);
    $app = $stmt->fetch();
    if (!$app) {
        http_response_code(404);
        echo 'Appointment not found.';
        exit;
    }

    // Check if already submitted
    $stmt = $pdo->prepare('SELECT id FROM liability_forms WHERE appointment_id = ?');
    $stmt->execute([$a]);
    if ($stmt->fetch()) {
        echo '<h2>Form already submitted.</h2>';
        exit;
    }

    // Render simple HTML form
    $fullName = htmlspecialchars($app['first_name'].' '.$app['last_name']);
    $apptTime = htmlspecialchars($app['scheduled_start'] . ' - ' . $app['scheduled_end']);
    $email = htmlspecialchars($app['email']);
    $phone = htmlspecialchars($app['phone']);

    echo '<!doctype html><html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Liability Form</title>
    <style>body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;margin:0;padding:16px;background:#f7f7f8} .card{max-width:780px;margin:0 auto;background:#fff;border-radius:12px;padding:20px;box-shadow:0 2px 8px rgba(0,0,0,.06)} .row{display:flex;gap:12px} .row>div{flex:1} label{display:block;margin:12px 0 6px;font-weight:600} input,select,textarea{width:100%;padding:10px;border:1px solid #ccc;border-radius:8px;font-size:14px} button{background:#0d6efd;color:#fff;border:none;border-radius:8px;padding:12px 16px;font-size:16px;margin-top:16px;cursor:pointer} .muted{color:#666;font-size:13px}</style>
    </head><body><div class="card">';
    echo '<h2>Repair Liability Form</h2>';
    echo '<p class="muted">Customer: '.$fullName.' | Email: '.$email.' | Phone: '.$phone.'<br/>Appointment: '.$apptTime.'</p>';
    echo '<form method="post" action="/submit">';
    echo '<input type="hidden" name="appointment_id" value="'.(int)$app['appointment_id'].'" />';
    echo '<div class="row"><div><label>Device Make</label><input name="device_make" required></div><div><label>Device Model</label><input name="device_model" required></div></div>';
    echo '<label>Device Password/Passcode (optional)</label><input name="device_password" placeholder="Optional">';
    echo '<label>Problem Description</label><textarea name="problem_description" rows="4" required></textarea>';
    echo '<label>Accessories Left with Device</label><input name="accessories_left" placeholder="e.g., charger, case, SIM">';
    echo '<label>Condition Notes</label><textarea name="condition_notes" rows="3"></textarea>';
    echo '<label>Contact Preference</label><select name="contact_preference"><option value="contact_ok">Contact OK</option><option value="no_contact">No Contact Drop-off</option></select>';
    echo '<label><input type="checkbox" name="consent_data_backup" required> I understand that while reasonable care is taken, data loss can occur and I am responsible for my own backups.</label>';
    echo '<label><input type="checkbox" name="consent_password_handling" required> I authorize handling of device security features (e.g., passcodes) as needed for diagnosis and repair.</label>';
    echo '<label>Type Your Full Legal Name (Signature)</label><input name="signature_name" required placeholder="Full name">';
    echo '<button type="submit">Submit Liability Form</button>';
    echo '</form></div></body></html>';
    exit;
}

http_response_code(404);
 echo 'Not Found';
